<?PHP
/*
 * Persistenz-Schicht
 * 
 * Die Persistenz-Schicht dient zum Speichern, Laden und Löschen von ToDos in der Datenbank. 
 * Sie besteht aus zwei Komponenten: dem sogenannten Resource Model und dem Repository. 
 * Das Resource Model kümmert sich um die Verwaltung eines einzelnen Domain Models, 
 * also ein Objekt von der Klasse TodoItem, 
 * das Repository kümmert sich um die Verwaltung aller Resourcen.
 * 
 * Repository
 * 
 * Das Repository repräsentiert eine Datensatz-Sammlung, es kapselt den Zugriff auf die Resourcen. 
 * Mit dem Repository können wir neue Datensätze erzeugen, 
 * eine spezifische Resource auslesen oder alle Resourcen nacheinander auslesen. 
 * Dafür hat unsere Repository-Klasse die drei Methoden get, all und add. 
 * Wir erklären im Anschluss die beiden Methoden all und add.
 *  
 * Die get-Methode ist etwas leichter zu verstehen und wird deshalb nicht erklärt.
 * 
 * Die all-Methode liest zunächst alle Datensatz-IDs aus der Datenbank aus. 
 * Dann iterieren wir in einer foreach-Schleife über alle Ergebnis-Zeilen der SQL-Anfrage. 
 * In jedem Schleifendurchlauf erstellen wir eine TodoItemResource-Instanz, 
 * die den Datensatz mit der entsprechenden ID repräsentiert. 
 * Wir möchten eine Liste dieser Instanzen zurückgeben, das machen wir, 
 * indem wir das Keyword yield vor jeder Insanz notieren. 
 * Das Ergebnis der Methode ist dann ein Iterator, über den man mit einer foreach-Schleife iterieren kann.
 * 
 * Die add-Methode funktioniert ähnlich wie die update-Methode der TodoItemResource-Klasse. 
 * Im Unterschied dazu gibt add aber eine TodoItemResource zurück, die den neu angelegten Datensatz repräsentiert. 
 * Dafür brauchen wir die lastInsertId-Methode von unserer Datenbank-Verbindung. 
 */
declare(strict_types=1);

namespace SelfHtml\Todo\Persistence;

use \PDO;
use \Iterator;
use \SelfHtml\Todo\Domain\TodoItem;

/**
 * Ein TodoItemRepository verwaltet den Zugriff auf TodoItemResources.
 */
final class TodoItemRepository
{
    /**
     * @var PDO
     */
    private $pdo;

    /**
     * Erzeugt ein Repository mit der angegebenen Datenbank-Verbindung
     *
     * @param PDO $pdo Datenbank-Verbindung
     */
    public function __construct(PDO $pdo)
    {
        $this->pdo = $pdo;
    }

    /**
     * Gibt eine Resource zurück, die den Datensatz mit der angegebenen Id repräsentiert.
     *
     * @param int $id Datensatz-ID
     * @return TodoItemResource
     */
    public function get(int $id) : TodoItemResource
    {
        return new TodoItemResource($this->pdo, $id);
    }

    /**
     * Gibt eine Liste aller Resourcen zurück.
     *
     * @return Iterator
     */
    public function all() : Iterator
    {
        static $query = 'SELECT `id` FROM `todo`';
        $statement = $this->pdo->prepare($query);
        $result = $statement->execute();
        foreach ($statement as $row) {
            yield new TodoItemResource($this->pdo, (int) $row['id']);
        }
    }

    /**
     * Erzeugt einen neuen Datensatz, der das übergebene Todo-Item repräsentiert.
     * Gibt eine Resource zurück, die den neuen Datensatz repräsentiert.
     *
     * @param TodoItem $todoItem Initialer Inhalt des neuen Datensatzes
     * @return TodoItemResource Resource, die den neuen Datensatz repräsentiert.
     */
    public function add(TodoItem $todoItem) : TodoItemResource
    {
        static $query = <<<SQL
            INSERT INTO `todo`
            SET `state` = :state, `description` = :description
            SQL;
        $statement = $this->pdo->prepare($query);
        $result = $statement->execute([
            ':state' => $todoItem->getState(),
            ':description' => $todoItem->getDescription()
        ]);
        return new TodoItemResource($this->pdo, (int) $this->pdo->lastInsertId());
    }
}
