<?php
/*
 * Controller
 * 
 * Endlich haben wir alle Komponenten, die wir brauchen, um uns dem Controller zuzuwenden. 
 * Der Controller soll die Formulardaten auswerten. 
 * Er untersucht sie darauf, ob ein neues ToDo eingegeben wurde und ob bestehende ToDos bearbeitet oder gelöscht wurden. 
 * Für jede dieser Aktionen ruft der Controller dann die Persistenz-Schicht auf, um die Änderungen in der Datenbank wirksam zu machen. 
 * Im letzten Schritt benutzt er den View, um die Ausgabe vorzubereiten.
 * Für die einzelnen Unteraufgaben hat unser Controller vier private Hilfsmethoden: 
 * create, update, delete und showForm. 
 * Die einzige öffentliche Methode indexAction führt die Gesamtaufgabe aus, indem sie die einzelnen Hilfsmethoden aufruft. 
 * Wir erklären im Folgenden nur die update-Methode.
 *
 * update-Methode
 * 
 * Die update-Methode bekommt in dem Parameter $updates alle Formular-Felder als Array übergeben, 
 * deren Name mit update beginnt. Die Struktur des Arrays ergibt sich aus unserem HTML-Formular. 
 * Das Array könnte zum Beispiel so aussehen:
 * [
 *  '3' => ['description' => 'Frühstücken', 'state' => 1],
 *  '5' => ['description' => 'Mittag essen', 'state' => 1]
 * ]
 * Wobei die Schlüssel auf oberster Array-Ebene den Datensatz-IDs entsprechen. 
 * Wir iterieren deshalb in einer foreach-Schleife über alle Einträge der Liste. 
 * In der Schleife erzeugen wir uns jeweils ein Objekt von der Klasse TodoItem, das die geänderte Aufgabe repräsentiert. 
 * Im Anschluss holen wir uns die Ressource, die zu der Datensatz-ID gehört, und rufen die update-Methode auf der Ressource auf. 
 * Bei diesem Schritt müssen wir uns wieder daran erinnern, dass das Bearbeiten schief gehen kann, falls der Datensatz nicht mehr existiert. 
 * In diesem Fall soll unsere Methode den Fehler abfangen und ignorieren. 
 */
declare(strict_types=1);

namespace SelfHtml\Todo\Controller;

use \OutOfBoundsException;
use SelfHtml\Todo\Domain\TodoItem;
use SelfHtml\Todo\Persistence\TodoItemRepository;
use SelfHtml\Todo\View\TodoListView;

/**
 * Der TodoController ist verantwortlich dafür, Formulardaten auszuwerten, entsprechende
 * Änderungen in der Persitenz-Schicht auszulösen und das Bearbeitungs-Formular zu erzeugen.
 */
final class TodoController
{
    /**
     * @var TodoItemRepository
     */
    private $repository;

    /**
     * @var TodoListView
     */
    private $view;

    public function __construct(TodoItemRepository $repository, TodoListView $view)
    {
        $this->repository = $repository;
        $this->view = $view;
    }

    /**
     * Wertet die Formulardaten aus, benutzt die Persistenz-Schicht zum Speichern der Änderungen
     * und die Darstellungsschicht, um das Formular zu erzeugen.
     *
     * @param array $formData Die Formulardaten aus dem $_POST-Array
     * @return string das HTML-Formular zum Bearbeiten der Todo-Liste
     */
    public function indexAction(array $formData) : string
    {
        $this->create($formData['create'] ?? []);
        $this->update($formData['update'] ?? []);
        $this->delete($formData['delete'] ?? []);
        return $this->showForm();
    }

    /**
     * Erstellt einen neuen Todo-Datensatz.
     *
     * @param array $newTodo Beschreibung eines Datensatzes
     */
    private function create(array $newTodo) : void
    {
        if (isset($newTodo['description']) && $newTodo['description'] !== '') {
            $item = new TodoItem(TodoItem::STATE_PENDING, $newTodo['description']);
            $this->repository->add($item);
        }
    }

    /**
     * Überschreibt Datensätze.
     *
     * @param array $updates
     */
    private function update(array $updates) : void
    {
        foreach ($updates as $id => $update) {
            $state = isset($update['state']) ? (TodoItem::STATE_DONE) : (TodoItem::STATE_PENDING);
            $item = new TodoItem($state, $update['description']);
            $resource = $this->repository->get((int) $id);
            try {
                $resource->update($item);
            } catch (OutOfBoundsException $e) {
            }
        }
    }

    /**
     * Löscht Datensätze.
     *
     * @param array $deletions
     */
    private function delete(array $deletions) : void
    {
        foreach ($deletions as $id => $deletion) {
            $resource = $this->repository->get((int) $id);
            try {
                $resource->delete();
            } catch (OutOfBoundsException $e) {
            }
        }
    }

    /**
     * Gibt das HTML-Formular zurück, das zum Neuanlegen, Bearbeiten und Löschen der Todo-Items dient.
     * 
     * @return string
     */
    private function showForm() : string
    {
        $resources = $this->repository->all();
        return $this->view->render($resources);
    }
}

