<?php
/* Das Resource Model repräsentiert einen Datensatz in der Datenbank. 
 * Es ist dafür verantwortlich, ein TodoItem aus dem Datensatz auszulesen, 
 * den Datensatz mit einem aktualisierten TodoItem zu überschreiben und den Datensatz zu löschen. 
 * Für die drei Operationen hat die folgende Klasse die drei korrespondierenden Methoden fetch, update und delete. 
 * Weil die Methoden sich sehr ähneln, erklären wir im Anschluss nur die update-Methode genauer, 
 * du solltest aber versuchen, alle Methoden zu verstehen. 
 * Wir haben zu jeder Methode PHP-Kommentare geschrieben, die dir beim Verstehen helfen sollen.
 * 
 * update-Methode
 *  
 * Die update-Methode bekommt ein TodoItem als Parameter übergeben, 
 * sie soll den entsprechenden Datensatz damit überschreiben. Zurückgeben soll die Methode nichts. 
 * In der Variablen $query speichern wir das Grundgerüst der SQL-Abfrage, 
 * die wir für die Änderung in der Datenbank brauchen. 
 * Die SQL-Vorlage enthält die Platzhalter :state und :description, 
 * die stellvertretend für den Zustand und die Beschreibung eines ToDos stehen 
 * und einen dritten Platzhalter :id für die Datensatz-ID. 
 * In der darauf folgenden Zeile wird aus dem Grundgerüst ein Datenbank-Statement erzeugt. 
 * Erst im dritten Schritt werden mit execute die tatsächlichen Werte für die Platzhalter eingesetzt 
 * und der Schreibvorgang durchgeführt. Dieser Schritt kann schief gehen, 
 * wenn der Datensatz in der Zwischenzeit gelöscht wurde. 
 * Wir fragen deshalb mit rowCount ab, wieviele Datensätze von der Update-Anfrage betroffen waren. 
 * Sollte kein Datensatz betroffen sein, melden wir eine OutOfBoundsException.
 */
declare(strict_types=1);

namespace SelfHtml\Todo\Persistence;

use \PDO;
use \OutOfBoundsException;
use \SelfHtml\Todo\Domain\TodoItem;

/**
 * Eine TodoItemResource repräsentiert einen existierenden Datensatz, der ein TodoItem enthält.
 * Der Datensatz wird durch eine eindeutige ID identifiziert, in diesem Fall ist
 * das eine von SQL generierte ID. Die TodoItemResource ist dafür verantwortlich, den Datensatz
 * auszulesen, ihn zu überschreiben und zu löschen.
 */
final class TodoItemResource
{
    /**
     * @var PDO
     */
    private $pdo;

    /**
     * @var int
     */
    private $id;

    /**
     * Erzeugt eine Resource mit der angegebenen Datenbank-Verbindung und Datensatz-ID.
     * Der Konstruktur erstellt keinen neuen Datensatz in der Datenbank, er muss
     * stattdessen mit der ID eines bereits existierenden Datensatzes aufgerufen werden.
     * Für die Erstellung neuer Datensätze ist das Repository verantwortlich.
     *
     * @param PDO $pdo Datenbank-Verbindung
     * @param int $id Datensatz-ID
     */
    public function __construct(PDO $pdo, int $id)
    {
        $this->pdo = $pdo;
        $this->id = $id;
    }

    /**
     * Gibt die Datensatz-ID der Ressource zurück.
     *
     * @return int
     */
    public function getId() : int
    {
        return $this->id;
    }

    /**
     * Liest das TodoItem aus dem Datensatz aus.
     *
     * @throws OutOfBoundsException falls der zu lesende Datensatz nicht existiert.
     *
     * @return TodoItem
     */
    public function fetch() : TodoItem
    {
        static $query = 'SELECT `state`, `description` FROM `todo` WHERE `id` = :id';
        $statement = $this->pdo->prepare($query);
        $result = $statement->execute([':id' => $this->id]);
        if ($statement->rowCount() === 0) {
            throw new OutOfBoundsException();
        }
        $row = $statement->fetch(PDO::FETCH_ASSOC);
        $todoItem = new TodoItem((int) $row['state'], $row['description']);
        return $todoItem;
    }

    /**
     * Überschreibt den Datensatz mit einem neuen TodoItem.
     *
     * @throws OutOfBoundsException falls der zu aktualisierende Datensatz nicht existiert.
     *
     * @param TodoItem neuer Datensatz
     */
    public function update(TodoItem $todoItem) : void
    {
        static $query = <<<SQL
            UPDATE `todo`
            SET `state` = :state, `description` = :description
            WHERE `id` = :id
            SQL;
        $statement = $this->pdo->prepare($query);
        $statement->execute([
            ':id' => $this->id,
            ':state' => $todoItem->getState(),
            ':description' => $todoItem->getDescription()
        ]);
        if ($statement->rowCount() === 0) {
            throw new OutOfBoundsException();
        }
    }

    /**
     * Löscht den Datensatz.
     *
     * @throws OutOufBoundsException falls der zu löschende Datensatz nicht existiert.
     */
    public function delete() : void
    {
        static $query = <<<SQL
            DELETE FROM `todo`
            WHERE `id` = :id
            SQL;
        $statement = $this->pdo->prepare($query);
        $statement->execute([
            ':id' => $this->id
        ]);
        if ($statement->rowCount() === 0) {
            throw new OutOfBoundsException();
        }
    }
}
