<?php
/* Domain_Model
 * Ein ToDo besteht aus einer textuellen Beschreibung der Aufgabe und einem Zustand, 
 * der entweder "ausstehend" oder "erledigt" sein kann. 
 * Wir definieren dafür die Klasse TodoItem in der Datei src/Domain/TodoItem.php. 
 * Die möglichen Zustände definieren wir als Klassen-Konstanten, 
 * weil ein Name wie TodoItem::STATE_PENDING aussagekräftiger als die Zahl 0 ist. 
 * Das verbessert die Lesbarkeit unseres Quelltextes.
 */
declare(strict_types=1);

namespace SelfHtml\Todo\Domain;

/**
 * Ein TodoItem repräsentiert eine Aufgabe. 
 * Zu einer Aufgabe gehört eine Beschreibung und ein Zustand,
 * der anzeigt, ob die Aufgabe bereits erledigt wurde oder nicht.
 */
final class TodoItem
{
    /**
     * @var int
     */
    private $state;

    /**
     * @var string
     */
    private $description;

    /**
     * Repräsentiert den Zustand einer noch nicht erfüllten Aufgabe.
     *
     * @var int
     */
    public const STATE_PENDING = 0;

    /**
     * Repräsentiert den Zustand einer erledigten Aufgabe.
     *
     * @var int
     */
    public const STATE_DONE = 1;

    /**
     * Erzeugt eine Aufgabe mit dem übergebenen Zustand und textueller Beschreibung.
     *
     * @param int $state Zustand der Aufgabe
     * @param string $description textuelle Beschreibung der Aufgabe
     */
    public function __construct(int $state, string $description)
    {
        $this->state = $state;
        $this->description = $description;
    }

    /**
     * Gibt den Zustand der Aufgabe zurück.
     *
     * @return int
     */
    public function getState() : int
    {
        return $this->state;
    }

    /**
     * Gibt die textuelle Beschreibung der Aufgabe zurück.
     *
     * @return string
     */
    public function getDescription() : string
    {
        return $this->description;
    }
}
