<?php
/*
 * Darstellungs-Schicht
 * 
 * Als nächstes kümmern wir uns um die Ausgabe in HTML. 
 * Wir werden nur ein einziges Formular brauchen, 
 * das gleichzeitig dem Erstellen, Bearbeiten und Löschen von Einträgen unserer ToDo-Liste dient. 
 * Für das Bearbeiten und Löschen benutzen wir eine private Hilfsmethode mit dem Namen renderResource, 
 * die wir in einer Schleife benutzen, um die Formularfelder für die existierenden Todos zu erzeugen.
 * 
 * Wir müssen daran denken, dass die fetch-Methode fehlschlagen kann, wenn ein Datensatz nicht mehr existiert. 
 * In der Hilfsmethode benutzen wir deshalb einen try/catch-Block, um den Fehler abzufangen. 
 * Wenn das passiert, dann verlassen wir die Hilfsfunktion vorzeitig und geben schlicht den leeren String zurück.
 */ 
declare(strict_types=1);

namespace SelfHtml\Todo\View;

use \Iterator;
use SelfHtml\Todo\Domain\TodoItem;
use SelfHtml\Todo\Persistence\TodoItemResource;

/**
 * Repräsentiert die Darstellung einer Todo-Liste.
 */
final class TodoListView
{
    /**
     * Erzeugt ein HTML-Formular zum Erstellen, Bearbeiten und Löschen von Todos.
     *
     * @param Iterator $rsources Bereits vorhandene Todos
     * @return string HTML-String
     */
    public function render(Iterator $resources) : string
    {
        $editsHtml = "";
        foreach ($resources as $resource) {
            $editsHtml .= $this->renderResource($resource);
        }
        return <<<HTML
            <!DOCTYPE html>
            <html lang="de">
                <head>
                    <title>SelfHtml TodoApp</title>
                </head>
                <body>
                    <h1>Todo Liste</h1>
                    <form method="POST">
                        $editsHtml
                        <fieldset>
                            <legend>Neuer Eintrag</legend>
                            <label>
                                <span>Beschreibung</span>
                                <input name="create[description]" maxlength="255">
                            </label>
                        </fieldset>
                        <button>Speichern</button>
                    </form>
                </body>
            </html>
            HTML;
    }

    /**
     * Erzeugt ein Teilformular zum Bearbeiten und Löschen eines existierendes Todos.
     *
     * @param TodoItemResource
     * @return string
     */
    private function renderResource(TodoItemResource $resource) : string
    {
        try {
            $item = $resource->fetch();
        } catch (OutOfBoundsException $e) {
            return '';
        }
        $state = $item->getState();
        $checked = ($state === TodoItem::STATE_DONE) ? 'checked' : '';
        $id = \htmlspecialchars((string) $resource->getId());
        $description = \htmlspecialchars($item->getDescription());
        return <<<HTML
            <fieldset>
                <legend>Eintrag bearbeiten</legend>
                <label>
                    <span>Beschreibung</span>
                    <input name="update[$id][description]" value="$description"  maxlength="255">
                </label>
                <label>
                    <span>Erledigt</span>
                    <input name="update[$id][state]" type="checkbox" $checked>
                </label>
                <label>
                    <span>Löschen</span>
                    <input name="delete[$id]" type="checkbox">
                </label>
            </fieldset>
            HTML;
    }
}

